/************************************************************************
 * NAME:	apple.c
 *
 * DESCR:	Support for apple 2 disk images.
 *
 * NOTES:	companion file appleutil.[c] contains utility routines.
 *		Also, gcr*.[ch] files contain GCR'ing code.
 *
 * MORE IMPORTANT NOTES:
 *	** Wed Apr 27 13:22:14 2005 **
 *	The state of this code as of today includes:
 *	   - AGCR6x2 encoding works well - that is, when the code sees a disk
 *		that isn't in nib format (sector based images) and appears
 *		to be a 3.3 image, it will be encoded in 6x2 and sent to
 *		the SVD.  This encoding simply rearranges the bits in the
 *		method that the ROM code uses.  The SVD will take those
 *		6-bit-oriented nibbles and convert them via GCR to the
 *		data that will appear as disk data.  Note that in this
 *		mode there are a few bits that don't fit in a standard
 *		256-byte sector, and are put into the header instead.
 *	   - AGCR5x3 is partially implemented - what should happen is that
 *		when a 3.2 image is recognized it "could" be encoded in
 *		5x3 and sent to the SVD, which would then run it through
 *		5x3 GCR encoding before being generated as disk data.
 *		The code on the SVD that does this is partially implemented
 *		as well...though it is commented out.  There needs to be
 *		quite a bit of work in appleutil.c for this to work too.
 *	   - NIB - nib format is implemented, and will take a nib-formated
 *		diskette and convert it BACK to AGCR6x2 data by reading
 *		the "bits" and doing the conversion.  This AGCR6x2 is then
 *		sent to the SVD as described above.  You can argue, after
 *		reading about RNIB (below) that this isn't too terribly
 *		useful.  One note, though, if the NIB is to survive, it 
 *		should only convert to AGCR6x2 if it _can_ convert to it.
 *		That is, the encoding should be checked to see if there
 *		is any difference between what would otherwise be generated
 *		with 6x2 and what is in the nib file.  If there IS a
 *		difference, then RNIB should be used...otherwise 6x2.
 *	   - RNIB - this is the raw nib format.  The "raw" part refers to
 *		the fact that this format is written to the SVD in a raw
 *		form that is just generated verbatim to the apple from the
 *		SVD.  The big difference between RNIB and the other formats
 *		is that RNIB takes more memory space.  HOWEVER, on the v1.0
 *		of the SVD, memory is only 256k and only one RNIB or one of
 *		the other formats will fit anyway.  RNIB supports NIB files
 *		that could be, for example, GCR5x3.  In fact, I can't find
 *		a file that is sector-based that is supposed to be GCR5x3.
 *		This argues strongly that GCR5x3 form is unnecessary.
 *
 *	Changes being made today include basically causing the RNIB format
 *	to be used by default.  NIB can be used by specifying it, though
 *	its utility is relatively low.
 *
 *	Things left to be done as of today include implementation of "write"
 *	in the SVD as well as in this code.  The question remains as to
 *	whether or not AGCR6x2 continues to be supported.  You can imagine
 *	that all sector-based 6x2 images could easily be converted to RNIB
 *	with code that already exists in these files.
 *
 *	Other formats that I was thinking of supporting include:
 *
 *	2MG - a 2MG header on raw sector data - maybe later!
 *
 *	Currently, too, there is no attempt to guess either PO/DO for a
 *	.dsk file.  With the ADOS file support, however, it could be done.
 *
 ************************************************************************/
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>

#include "floppy.h"
#include "error.h"
#include "standard.h"
#include "apple.h"

/************************************************************************
 * NAME:	do_guesser()
 *
 * DESCR:	Does a guess at the DO format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- not much happens here, except that the size of the file
 *		  is checked.
 *		- eventually, this needs to try to differentiate between
 *		  the two different types of ordering.
 *		- this routine takes advantage of the fact that very few
 *		  if ANY other formats have such a static (odd) size.
 ************************************************************************/
int
do_guesser(int fd)
{
    struct stat		statbuf;

    fstat(fd,&statbuf);

    if (statbuf.st_size != (256*16*35)) {	/* either 16 sector 6x2	*/
	if (statbuf.st_size != (256*13*35)) {	/*  or 13 sector 5x3	*/
	    return(FALSE);
	}
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	po_guesser()
 *
 * DESCR:	Does a guess at the PO format.  Note that this is EXACTLY
 *		the same as the DO guessor.  And since this is behind the
 *		DO guessor, this routine probably won't ever be called!
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- not much happens here, except that the size of the file
 *		  is checked.
 *		- eventually, this needs to try to differentiate between
 *		  the two different types of ordering.
 *		- this routine takes advantage of the fact that very few
 *		  if ANY other formats have such a static (odd) size.
 ************************************************************************/
int
po_guesser(int fd)
{
    struct stat		statbuf;

    fstat(fd,&statbuf);

    if (statbuf.st_size != (256*16*35)) {	/* either 16 sector 6x2	*/
	if (statbuf.st_size != (256*13*35)) {	/*  or 13 sector 5x3	*/
	    return(FALSE);
	}
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	raw_read()
 *
 * DESCR:	Given a PO or DO format, read it in as raw data.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- WAY simple
 *		- doesn't even contemplate double sided!
 ************************************************************************/
static int
raw_read(int fd, struct floppy *floppy, int *order)
{
    int		track;
    int 	sector_length = 256;	/* should move this to floppy	*/
    int		sector;
    struct stat	statbuf;

    fstat(fd,&statbuf);

#define THIS_SECTOR	floppy->side[0][track].sector[sector]

    floppy->tracks = 35;
    floppy->sectors = statbuf.st_size / sector_length / floppy->tracks;
    floppy->write_protect = 0;		/* not write-protected		*/
    floppy->sides = 1;
    floppy->secsize = 256;

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));

    for (track = 0; track < floppy->tracks; track++) {
	floppy->side[0][track].sector = 
	    (struct sector *)malloc(floppy->sectors*sizeof(struct sector));
	floppy->side[0][track].sectors = floppy->sectors;
    }

    for(track =0; track < floppy->tracks; track++) {
	int	count;

	for (sector = 0; sector < floppy->sectors; sector++) {

	    count = read(fd,THIS_SECTOR.data,(size_t)sector_length);

	    if (count < sector_length) {
		break;	/* must be done, didn't get a complete track*/
	    }

	    /* here are the apple format specific values	*/

	    THIS_SECTOR.encoding = (floppy->tracks==13)?AGCR5x3:AGCR6x2;
	    floppy->encoding = THIS_SECTOR.encoding;

	    THIS_SECTOR.id = track;
	    THIS_SECTOR.sector = order[sector];
	    THIS_SECTOR.volume = 0xfe;
	    THIS_SECTOR.hdrchecksum = THIS_SECTOR.volume ^ THIS_SECTOR.id ^ THIS_SECTOR.sector;


	    THIS_SECTOR.hdrprolog[0] = 0xd5;
	    THIS_SECTOR.hdrprolog[1] = 0xaa;
	    if (THIS_SECTOR.encoding == AGCR6x2) {
		THIS_SECTOR.hdrprolog[2] = 0x96;
	    } else {
		THIS_SECTOR.hdrprolog[2] = 0xb5;
	    }
	    THIS_SECTOR.dataprolog[0] = 0xd5;
	    THIS_SECTOR.dataprolog[1] = 0xaa;
	    THIS_SECTOR.dataprolog[2] = 0xad;

	    THIS_SECTOR.hdrepilog[0] = 0xde;
	    THIS_SECTOR.hdrepilog[1] = 0xaa;
	    THIS_SECTOR.hdrepilog[2] = 0xeb;

	    THIS_SECTOR.dataepilog[0] = 0xde;
	    THIS_SECTOR.dataepilog[1] = 0xaa;
	    THIS_SECTOR.dataepilog[2] = 0xeb;

	    {
		/* need to calculate the checksum here	*/

		unsigned char	buffer[410];	/* big enuf for 5x3 (and 6x2) */
		unsigned char	preload;

		if (THIS_SECTOR.encoding == AGCR6x2) {
		    
		    THIS_SECTOR.datachecksum = 
			a2_6x2_gcr_translate(THIS_SECTOR.data, buffer);

		} else {

		    THIS_SECTOR.datachecksum = 
			a2_5x3_gcr_translate(THIS_SECTOR.data, buffer);
		}
	    }

	    /* NOTE - the "preload" isn't handled here in that it is logically	*/
	    /*  an issue only for the SVD format and is handled by it.		*/

	    /* here's the old stuff that is just filled in for fun	*/

	    THIS_SECTOR.side = 0;
	    THIS_SECTOR.size = 256;
	    THIS_SECTOR.mark = 0;
	    THIS_SECTOR.headCRC = 0;
	    THIS_SECTOR.dataCRC = 0;
	}
    }

    floppy->tracks = track;

    /* need to compute the sector map ahead of time so interleaving will work	*/

    floppy_sectormap(floppy);
    floppy_interleave(floppy,1);

    return(0);
}

int DOS_ORDER[] = { 0, 13, 11, 9, 7, 5, 3, 1, 14, 12, 10, 8, 6, 4, 2, 15 };
int PRODOS_ORDER[] = { 0, 2, 4, 6, 8, 10, 12, 14, 1, 3, 5, 7, 9, 11, 13, 15 };

/************************************************************************
 * NAME:	do_read()
 *
 * DESCR:	Read in DO format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
do_read(int fd, struct floppy *floppy)
{
    return(raw_read(fd,floppy,DOS_ORDER));
}

/************************************************************************
 * NAME:	po_read()
 *
 * DESCR:	Read in PO format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
po_read(int fd, struct floppy *floppy)
{
    return(raw_read(fd,floppy,PRODOS_ORDER));
}

/************************************************************************
 * NAME:	do_report()
 *
 * DESCR:	
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
do_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("DO");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = do_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

/************************************************************************
 * NAME:	po_report()
 *
 * DESCR:	
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
po_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("PO");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = po_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

/************************************************************************
 * NAME:	raw_dump()
 *
 * DESCR:	Dumps either DO or PO formats.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
raw_dump(int fd, struct floppy *floppy, int *order)
{
}

/************************************************************************
 * NAME:	do_dump()
 *
 * DESCR:	
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
do_dump(int fd, struct floppy *floppy)
{
    return(raw_dump(fd,floppy,DOS_ORDER));
}

/************************************************************************
 * NAME:	po_dump()
 *
 * DESCR:	
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
po_dump(int fd, struct floppy *floppy)
{
    return(raw_dump(fd,floppy,PRODOS_ORDER));
}

/************************************************************************
 * NAME:	do_init()
 *
 * DESCR:	Registers the do floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if registration went OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
do_init()
{
    if (!floppy_format_register("DO", "APPLE 2 DOS",do_guesser, do_read, NULL, do_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("do",EXT_VERY_SPECIFIC,"DO")) {
	return(FALSE);
    }
    if (!floppy_fileext_register("dsk",EXT_SOMEWHAT_SPECIFIC,"DO")) {
	return(FALSE);
    }
    
    return(TRUE);
}

/************************************************************************
 * NAME:	po_init()
 *
 * DESCR:	Registers the po floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if registration went OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
po_init()
{
    if (!floppy_format_register("PO", "APPLE 2 PRODOS",po_guesser, po_read, NULL, po_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("po",EXT_VERY_SPECIFIC,"PO")) {
	return(FALSE);
    }
    if (!floppy_fileext_register("dsk",EXT_SOMEWHAT_SPECIFIC,"PO")) {
	return(FALSE);
    }
    
    return(TRUE);
}

/************************************************************************
 * NAME:	nib_guesser()
 *
 * DESCR:	Guess the nib format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- simple guessing...looks for the first 500 bytes
 *		  with the first bit set.
 ************************************************************************/
int
nib_guesser(int fd)
{
    int			i;
    struct stat		statbuf;
    unsigned char	buffer[500];

    fstat(fd,&statbuf);

    if (statbuf.st_size < 500) {
	return(FALSE);
    }

    if (read(fd,buffer,500) < 500) {
	return(FALSE);
    }

    for (i=0; i < 500; i++) {
	if (buffer[i] < 0x80) {
	    return(FALSE);
	}
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	nib_read()
 *
 * DESCR:	Read the nib format.  The idea here is that we pretend to
 *		be a diskette, and interpret the data in the nib format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- NIB files are processed as follows:
 *		  - expect a sector separator BEFORE each sector
 *		  - if this separator is over a certain track-indicator
 *		    minimum, then interpret the separator as a TRACK
 *		    separator.
 *
 *		NIB_MIN_SECTOR_SEP - the separator before a sector header
 *		NIB_MIN_DATA_SEP - between the header and the data
 *		NIB_MIN_TRACK_SEP - after a track, before the next one
 ************************************************************************/
int
nib_read_old(int fd, struct floppy *floppy)
{

#define NIB_MIN_SECTOR_SEP	10
#define NIB_MIN_DATA_SEP	5
#define NIB_MIN_TRACK_SEP	50

/* the data size limit is used as the upper limit for data, including checksum
   and the first byte of the data epilog	*/

#define DATA_SIZE_LIMIT		412

#define	E1	0xde
#define	E2	0xaa
#define	E3	0xeb

#define GOOD_SECTOR_COUNT(count)	(count==13||count==16)
#define GOOD_DATA_SIZE(count,locked)	((locked!=0)?(count==locked):(count==411||count==343))
#define GCR_REVTRANSLATE(in,count,out,c)((count==411)?a2_5x3_gcr_revtranslate(in,out,c):a2_6x2_gcr_revtranslate(in,out,c))

#undef THIS_SECTOR
#define THIS_SECTOR	floppy->side[0][track].sector[psector]

#define ERROR	printf
#define WARNING	printf

    unsigned char	buffer[DATA_SIZE_LIMIT];	/* big enough for a 5x3 sector	*/
    unsigned char	sectbuffer[256];
    int			eof = FALSE;
    int			psector;
    int			track;
    int			data_size_locked = 0;
    int			count;
    int			i;

    psector = 0;
    track = 0;

    /* assume at least one track with at least one sector	*/

    floppy->sides = 1;
    floppy->write_protect = 0;
    floppy->secsize = 256;

    floppy->side[0] = (struct track *)malloc(sizeof(struct track));
    floppy->tracks = 1;
    floppy->side[0][0].sectors = 0;
    
    for (;; psector++) {	/* loop indefinitely	*/

	{    /* read in the sector separator	*/

	    /* first wait for a "self sync" 0xff	*/

	    for (;;) {
		if (read(fd,buffer,(size_t)1) != 1) {
		    eof = TRUE;
		    break;
		}
		if (buffer[0] == 0xff) {
		    break;
		}
	    }

	    for (count=1; ; count++) {
		if (read(fd,buffer,(size_t)1) != 1) {
		    eof = TRUE;
		    break;
		}

		if ( buffer[0] != 0xff) {
		    break;
		}
	    }

	    if (eof) {
		if (floppy->tracks != 35) {
		    WARNING("Non-standard number of tracks %d\n",floppy->tracks);
		}
		floppy->sectors = psector;
		return(0);
	    }

	    if (count > NIB_MIN_SECTOR_SEP) {	/* have (at least) a sector sep	*/
		if (count > NIB_MIN_TRACK_SEP) {/* looks to be end-of-track/sector sep	*/
		    if (!GOOD_SECTOR_COUNT(psector)) {
			ERROR("Bad number of sectors in track %d with %d sectors",track,psector);
			return(1);
		    }
		    track++;
		    floppy->tracks++;
		    floppy->side[0] = (struct track *)realloc(floppy->side[0], floppy->tracks * sizeof(struct track));
		    floppy->side[0][track].sector = (struct sector *)malloc(sizeof(struct sector));
		    floppy->side[0][track].sectors = 0;
		    psector = 0;
		}
	    } else {
		ERROR("Sector separator too short track %d sector %d\n",track,psector);
		return(1);
	    }
	}

	{	/* read the sector header (already have one byte)	*/

	    floppy->side[0][track].sectors++;
	    if (floppy->side[0][track].sectors == 1) {
		floppy->side[0][track].sector = (struct sector *)malloc(sizeof(struct sector));
	    } else {
		floppy->side[0][track].sector = (struct sector *)realloc(
		                                                    floppy->side[0][track].sector,
							            floppy->side[0][track].sectors*sizeof(struct sector));
	    }

	    if (read(fd,buffer+1,(size_t)13) != 13) {
		ERROR("EOF found during sector header at track %d sector %d\n",track,psector);
		return(1);
	    }
	    THIS_SECTOR.hdrprolog[0] = buffer[0];
	    THIS_SECTOR.hdrprolog[1] = buffer[1];
	    THIS_SECTOR.hdrprolog[2] = buffer[2];
	    for (i=3; i < 11; i++) {		/* quick check on header raw bytes	*/
		if (buffer[i] & 0xaa != 0xaa) {
		    ERROR("Header data malformed at track %d sector %d\n",track,psector);
		    return(1);
		}
	    }
	    /* NEED TO SET THESE ACCORDING TO HEADER in FLOPPY!  BUG! BUG! BUG! */
	    floppy->encoding = AGCR6x2;
	    THIS_SECTOR.encoding = AGCR6x2;
	    THIS_SECTOR.mark = 0;
	    THIS_SECTOR.headCRC = 0;
	    THIS_SECTOR.dataCRC = 0;
	    THIS_SECTOR.side = 0;
	    THIS_SECTOR.size = 256;
    	    THIS_SECTOR.sizecode = 1;

/*	    M4("reading in vol, id, sec, cs: 0x%02x 0x%02x 0x%02x 0x%02x\n",
	       a2_gcr4x4_revmap(buffer[3],buffer[4]),
	       a2_gcr4x4_revmap(buffer[5],buffer[6]),
	       a2_gcr4x4_revmap(buffer[7],buffer[8]),
	       a2_gcr4x4_revmap(buffer[9],buffer[10])); */

	    THIS_SECTOR.volume = a2_gcr4x4_revmap(buffer[3],buffer[4]);
	    THIS_SECTOR.id = a2_gcr4x4_revmap(buffer[5],buffer[6]);
	    THIS_SECTOR.sector = a2_gcr4x4_revmap(buffer[7],buffer[8]);
	    THIS_SECTOR.hdrchecksum = a2_gcr4x4_revmap(buffer[9],buffer[10]);

	    THIS_SECTOR.hdrepilog[0] = buffer[11];
	    THIS_SECTOR.hdrepilog[1] = buffer[12];
	    THIS_SECTOR.hdrepilog[2] = buffer[13];
	}

	{	/* read the data separator	*/

	    for (count=0; ; count++) {
		if (read(fd,buffer,(size_t)1) != 1) {
		    ERROR("EOF encountered while moving to sector data at track %d sector %d\n",track,psector);
		    return(1);
		}

		if (buffer[0] != 0xff) {
		    break;
		}
	    }
	    if (count < NIB_MIN_DATA_SEP) {	/* have a data sep	*/
		ERROR("Data separator too short at track %d sector %d\n",track,psector);
		return(1);
	    }
	}

	{	/* read the sector data - unfortunately, done 1 byte at a time		*/

	    /* prolog first - already 1 byte read into this	*/

	    if (read(fd,buffer+1,(size_t)2) != 2) {
		ERROR("EOF found during data prolog at track %d sector %d\n",track,psector);
		return(1);
	    }
	    THIS_SECTOR.dataprolog[0] = buffer[0];
	    THIS_SECTOR.dataprolog[1] = buffer[1];
	    THIS_SECTOR.dataprolog[2] = buffer[2];

	    for(count=0; count < DATA_SIZE_LIMIT; count++) {
		if (read(fd,&buffer[count],(size_t)1) != 1) {
		    ERROR("EOF encountered during data read at track %d sector %d\n",track,psector);
		    return(1);
		}
		if (buffer[count] == E1 && GOOD_DATA_SIZE(count,data_size_locked)) {
			break;
		}
	    }

	    if (count == DATA_SIZE_LIMIT) {
		ERROR("Data epilog not encountered during data read at track %d sector %d\n",track,psector);
		return(1);
	    }

	    if (!GOOD_DATA_SIZE(count,data_size_locked)) {
		ERROR("Bad data size %d at track %d sector %d\n",count,track,psector);
		return(1);
	    }

	    /* note that the data checksum is NOT "checked"	*/

	    THIS_SECTOR.datachecksum = buffer[count-1];

	    if (!GCR_REVTRANSLATE(buffer,count,THIS_SECTOR.data,THIS_SECTOR.datachecksum)) {
		ERROR("Error in sector data at track %d sector %d\n",track,psector);
		return(1);
	    }

	    buffer[0] = buffer[count];	/* move the first byte of the epilog	*/

	    if (read(fd,buffer+1,(size_t)2) != 2) {
		ERROR("EOF encountered just after data read at track %d sector %d\n",track,psector);
		return(1);
	    }

	    THIS_SECTOR.dataepilog[0] = buffer[0];
	    THIS_SECTOR.dataepilog[1] = buffer[1];
	    THIS_SECTOR.dataepilog[2] = buffer[2];
	}
    }
}

/************************************************************************
 * NAME:	nib_find_address()
 *
 * DESCR:	Given a buffer of GCR data from a NIB file, find the first
 *		sector address based upon the DOS 3.3 search critereon.
 *
 * ARGS:	input - a pointer to input NIB/GCR data
 *		end - an index to the end of the input data (EXCLUSIVE)
 *		sector - the sector structure is filled in with header data
 *
 * RETURNS:	index within the buffer where the address started.  This
 *			index is directly to the first byte of the prolog.
 *
 * NOTES:	- 6x2	HEADER	- PROLOG - D5 AA 96*
 *		-		- EPILOG - DE AA EB
 *		-	DATA	- PROLOG - D5 AA AD
 *		-		- EPILOG - DE AA EB
 * 		- 5x3	HEADER	- PROLOG - D5 AA B5*
 *		-		- EPILOG - DE AA EB
 *		-	DATA	- PROLOG - D5 AA AD
 *		-		- EPILOG - DE AA EB
 ************************************************************************/
int
nib_find_address(unsigned char	*input,	/* input GCR data from NIB file	*/
		 int		 end,
		 int		*size,		/* amt of data consumed	*/
		 struct sector	*sector)
{
    unsigned char	b1, b2;
    int			start = -1;
    int			i;
    int			state = 0;

#define IS_GCR6x2	(sector->hdrprolog[2] == 0x96)
#define IS_GCR5x3	(sector->hdrprolog[2] == 0xB5)
#define DEFAULT_TYPE	AGCR6x2

    for (i=0; i < end; i++) {

	/* in the code below, case 9, the third byte of the prolog isn't checked
	   for any particular value.  It is just stored as part of the prolog.	*/

	switch(state) {
	    case 0:	/* fall thru */
	    case 1:	/* fall thru */
	    case 2:	/* fall thru */
	    case 3:	/* fall thru */
	    case 4:	/* fall thru */
	    case 5:	/* fall thru */
	    case 6:	if (input[i] == 0xff) { state++; } else { state = 0; } break;
	    case 7:	if (input[i] == 0xff) { break; } /* locks in at state 7 */
		        if (input[i] == 0xd5) { sector->hdrprolog[0] = input[i]; state++; } else { state = 0; } break;
	    case 8:	if (input[i] == 0xaa) { sector->hdrprolog[1] = input[i]; state++; } else { state = 0; } break;
/*	    case 9:	if (input[i] == 0x96) { sector->hdrprolog[2] = input[i]; state++; } else { state = 0; } break;	*/
	    case 9:	sector->hdrprolog[2] = input[i]; state++; break;
	   default:	break;
	}

	if (state == 10) {		/* located a data address	*/
	    i++;

	    if (end-i < 11) {
		break;			/* don't have enough data for header	*/
	    }
	    
	    start = i - 3;		/* set index to point to start of data	*/

	    b1 = input[i++]; b2 = input[i++]; sector->volume = a2_gcr4x4_revmap(b1,b2);
	    b1 = input[i++]; b2 = input[i++]; sector->id = a2_gcr4x4_revmap(b1,b2);
	    b1 = input[i++]; b2 = input[i++]; sector->sector = a2_gcr4x4_revmap(b1,b2);
	    b1 = input[i++]; b2 = input[i++]; sector->hdrchecksum = a2_gcr4x4_revmap(b1,b2);

	    sector->encoding = (IS_GCR6x2)?AGCR6x2:(IS_GCR5x3)?AGCR5x3:DEFAULT_TYPE;

/****
	    M2("sector encoding is %s (0x%02x)\n", (IS_GCR6x2)?"6x2":(IS_GCR5x3)?"5x3":"default", sector->hdrprolog[2]);
***/

	    sector->hdrepilog[0] = input[i++];
	    sector->hdrepilog[1] = input[i++];
	    sector->hdrepilog[2] = input[i++];

	    break;
	}
    }

    *size = i;
    return(start);
}

/************************************************************************
 * NAME:	nib_find_data()
 *
 * DESCR:	From the current position, find the sector data.  Return
 *		the CONVERTED data.
 *
 * ARGS:	
 *
 * RETURNS:	the index of the NEXT byte after the data was read, -1 if
 *			no data was found.
 *
 ************************************************************************/
int				       
nib_find_data(unsigned char	*input,
	      int		 end,
	      int		*size,		/* amt of data consumed	*/
	      struct sector	*sector)
{
    int			start = -1;
    int			i;
    int			state = 0;
    int			datasize;

#define GCR6x2_SIZE	342
#define GCR5x3_SIZE	410
#define DEFAULT_SIZE	GCR6x2_SIZE

    for (i=0; i < end; i++) {

	/* in the code below, case 6, the third byte of the prolog isn't checked
	   for any particular value.  It is just stored as part of the prolog.	*/

	switch(state) {
	    case 0:	/* fall thru */
	    case 1:	/* fall thru */
	    case 2:	/* fall thru */
	    case 3:	if (input[i] == 0xff) { state++; } else { state = 0; } break;
	    case 4:	if (input[i] == 0xff) { break; }
		        if (input[i] == 0xd5) { sector->dataprolog[0] = input[i]; state++; } else { state = 0; } break;
	    case 5:	if (input[i] == 0xaa) { sector->dataprolog[1] = input[i]; state++; } else { state = 0; } break;
/*****	    case 6:	if (input[i] == 0xad) { sector->dataprolog[2] = input[i]; state++; } else { state = 0; } break;	*/
	    case 6:	state++; sector->dataprolog[2] = input[i]; break;
	   default:	break;
	}

	if (state == 7) {		/* located the data 		*/
	    i++;

	    switch(sector->encoding) {
	        case AGCR5x3:	datasize = GCR5x3_SIZE; break;
	        case AGCR6x2:	datasize = GCR6x2_SIZE; break;
	        default:	datasize = DEFAULT_SIZE; break;
	    }

	    if (end-i < datasize+1+3) {
		break;			/* don't have enough data for sector	*/
	    }

	    start = i - 3;		/* set start to point to 0xd5	*/

	    sector->datachecksum = input[i+datasize];
/***
	    M1("incoming checksum is 0x%02x\n",sector->datachecksum);
***/
	    if (IS_GCR5x3) {
		if (!a2_5x3_gcr_revtranslate(&(input[i]),sector->data,sector->datachecksum)) {
		    M("bad checksum\n");
		}
	    } else {
		/* defaults to GCR6x2 	*/
		if (!a2_6x2_gcr_revtranslate(&(input[i]),sector->data,sector->datachecksum)) {
		    M("bad checksum");
		}
	    }

	    i += datasize;

	    i++;			/* skip over checksum */

	    sector->dataepilog[0] = input[i++];
	    sector->dataepilog[1] = input[i++];
	    sector->dataepilog[2] = input[i++];

	    break;
	}
    }

    *size = i;
    return(start);
}


/************************************************************************
 * NAME:	nib_sector_read()
 *
 * DESCR:	Reads in a sector from the data buffer given in "ptr".
 *		Returns the amount of data consumed in ptr.  Also, the
 *		*start is filled in with the index relative to ptr of
 *		where the data started.
 *
 * ARGS:	ptr  - data buffer
 *		end  - index within ptr that is the end of data (exclusive)
 *
 * RETURNS:	-1 means that a sector couldn't be found
 *
 * NOTES:	
 ************************************************************************/
int
nib_sector_read(unsigned char *ptr, int end, int *start, struct sector *sector)
{
    int		 size1, size2;

    sector->side = 0;
    sector->sizecode = 1;
    sector->size = 256;

    *start = nib_find_address(ptr, end, &size1, sector);

    if (*start == -1) {
	return(-1);		/* couldn't find a sector	*/
    }

    ptr += size1;
    end -= size1;

    if (nib_find_data(ptr, end, &size2, sector) == -1 ) {
	return(-1);			/* couldn't find data for sector	*/
    }

    return(size1+size2);
}

/************************************************************************
 * NAME:	nib_track_read()
 *
 * DESCR:	Reads a given track from NIB format.
 *
 * ARGS:	input - assumed to be NIB_TRACK_SIZE (6656) bytes of NIB data -
 *			alignment is completely unknown - that is, the buffer
 *			could start in the middle of a track.
 *		track - where the output data is to be written - it is
 *			assumed that the track has storage for 16 sectors
 *			(already allocated)
 *
 * RETURNS:	the count of the number of sectors found, zero if an
 *		error of great proportion is discovered.
 *
 * NOTES:	- 
 ************************************************************************/
#define NIB_TRACK_SIZE	6656

int
nib_track_read(unsigned char *input, struct track *track)
{
    unsigned char	 buffer[2*NIB_TRACK_SIZE];	/* wrap-around is done cheaply	*/
    unsigned char	*ptr;
    int			 start, size;
    int			 end;
    int			 first_sector = TRUE;
    int			 sector;
    struct sector	 sector_buffer;

    /* copy the data stream to its own tail to make wrap-around of the	*/
    /*   track brain-dead simple...though it takes more space.		*/

    memcpy(buffer,input,(size_t)NIB_TRACK_SIZE);
    memcpy(buffer+NIB_TRACK_SIZE,input,(size_t)NIB_TRACK_SIZE);

    end = NIB_TRACK_SIZE;			/* this is the end of the data	*/

    ptr = buffer;

    track->sectors = 0;
    track->sector = NULL;

    for (sector=0;;sector++) {

	size = nib_sector_read(ptr,end,&start,&sector_buffer);

	if (size == -1) {
	    break;				/* couldn't find a sector	*/
	}

	if (first_sector) {
	    ptr += start;
	    size -= start;
	    first_sector = FALSE;
	}

	ptr += size;
	end -= size;

	track->sectors++;
	track->sector = (struct sector *)realloc(track->sector,track->sectors*sizeof(struct sector));
	track->sector[sector] = sector_buffer;
/*****
	M2("got a sector with data:  track %d, sector %d\n", track->sector[sector].id, track->sector[sector].sector);
****/
    }

    return(sector);
}

/************************************************************************
 * NAME:	nib_read()
 *
 * DESCR:	Read in a nib-format file.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
nib_read(int fd, struct floppy *floppy)
{
    int		track, sectors;

    /* assume at least one track with at least one sector	*/

    floppy->sides = 1;
    floppy->write_protect = 0;
    floppy->secsize = 256;

    floppy->side[0] = NULL;
    floppy->tracks = 0;

    for (track = 0; ; track++) {

	unsigned char	buffer[NIB_TRACK_SIZE];
	int		size;

	size = read(fd,buffer,NIB_TRACK_SIZE);
	if (size != NIB_TRACK_SIZE) {			/* EOF reached	*/
	    if (size !=0) {
		M1("WARNING: EOF size read of %d (not zero)\n",size);
	    }
	    break;
	}

	floppy->tracks++;
	floppy->side[0] = (struct track *)realloc(floppy->side[0], floppy->tracks*sizeof(struct track));
	floppy->side[0][track].sectors = 0;

	sectors = nib_track_read(buffer, &(floppy->side[0][track]));
/***
	M2("track %d sectors %d\n",track,sectors);
***/
    }

    floppy->sectors = floppy->side[0][0].sectors;
    floppy->encoding = floppy->side[0][0].sector[0].encoding;

    return(0);
}


debug_print(unsigned char *buffer, int count)
{
    int	i;
    int crcount = 1;

    for(i=0; i < count; i++, crcount++) {
	printf("0x%02x ",buffer[i]);
	if (crcount % 8 == 0) {
	    printf("\n");
	}
	if (count > 256 && i == 85) {
	    printf("\n");
	    crcount = 0;
	}
    }

    printf("\n");
}
    


/************************************************************************
 * NAME:	nib_dump()
 *
 * DESCR:	
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
nib_dump(int fd, struct floppy *floppy)
{
}

/************************************************************************
 * NAME:	nib_report()
 *
 * DESCR:	
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
nib_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("NIB");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = nib_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

/************************************************************************
 * NAME:	nib_init()
 *
 * DESCR:	NIBble format for apple.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
nib_init()
{
    if(!floppy_format_register("NIB", "APPLE 2 NIBble",nib_guesser, nib_read, NULL, nib_report)) {
	return(FALSE);
    }

    /* the "SOMEWHAT" below will sort "nib" below "rnib" so rnib is checked first	*/

    if(!floppy_fileext_register("nib",EXT_SOMEWHAT_SPECIFIC,"NIB")) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	rnib_read()
 *
 * DESCR:	Read in a nib file in RAW format.  This is really a hack
 *		to make it possible to easily cause the SVD encoding of a
 *		nib file to be downloaded in a raw fashion to the SVD.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- many assumptions in the rnib format:
 *			- 35 tracks only
 *			- 6656 bytes per track only
 *			- file must be JUST right!
 *		
 ************************************************************************/
int
rnib_read(int fd, struct floppy *floppy)
{

    unsigned char	track_buffer[NIB_TRACK_SIZE];
    unsigned char	*ptr;
    int			track, sector;
    int			count;

#define RNIB_TRACKS	35
#define RNIB_SECTORS	26

    floppy->sides = 1;
    floppy->write_protect = 0;
    floppy->secsize = 256;
    floppy->encoding = RNIB;
    floppy->sectors = RNIB_SECTORS;

    floppy->tracks = RNIB_TRACKS;
    floppy->side[0] = (struct track *)malloc(floppy->tracks * sizeof(struct track));

    for (track=0; track < floppy->tracks; track++) {
	floppy->side[0][track].sectors = floppy->sectors;
	floppy->side[0][track].sector = 
	    (struct sector *)malloc(floppy->side[0][track].sectors * sizeof(struct sector));

	count = read(fd,track_buffer,NIB_TRACK_SIZE);
	if (count < NIB_TRACK_SIZE) {
	    return(1);
	}

	ptr = track_buffer;
	for (sector = 0; sector < floppy->side[0][track].sectors; sector++) {
	    memcpy(floppy->side[0][track].sector[sector].data,ptr,floppy->secsize);
	    ptr += floppy->secsize;
	    floppy->side[0][track].sector[sector].encoding = floppy->encoding;
	    floppy->side[0][track].sector[sector].id = track;
	    floppy->side[0][track].sector[sector].sector = sector;
	    floppy->side[0][track].sector[sector].side = 0;
	    floppy->side[0][track].sector[sector].size = floppy->secsize;
	    floppy->side[0][track].sector[sector].encoding = floppy->encoding;
	}
    }

    return(0);
}


rnib_dump(int fd, struct floppy *floppy) {};

rnib_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("RNIB");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = rnib_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

rnib_guesser(int fd)
{
    return(nib_guesser(fd));
};

/************************************************************************
 * NAME:	rnib_selfsync_encode()
 *
 * DESCR:	Used to encode self-syncs in an RNIB file.  Self-syncs
 *		are encoded as 0x00.
 *
 *		This routine is called after seeing a 0xd5 lead-in to
 *		a header block (where the self-sync's will be).
 *
 * ARGS:	floppy - the floppy
 *
 * RETURNS:	
 *
 * NOTES:	- we back up, up to 5 contiguous 0xff's to encode self-syncs
 *		- this means that we may need to back up a sector
 ************************************************************************/
void
rnib_selfsync_encode(struct floppy *floppy,	/* floppy struct to work on	*/
		     int	    track,	/* the track where we found d5	*/
		     int	    sector,	/* the sector where we found d5	*/
		     int	    d5_pos)	/* position of 0xd5 in data	*/
{
#define D5_BACKUP	5

    struct track	*tpr = &(floppy->side[0][track]);
    int			 count = D5_BACKUP;
    int			 pos = d5_pos;

    for (pos = d5_pos-1; pos >= 0 && count; pos--, count--) {
	if (tpr->sector[sector].data[pos] == 0xff) {
	    tpr->sector[sector].data[pos] = 0x00;
	} else {
	    return;			/* return on first non-contiguous 0xff	*/
	}
    }

    if( count ) {		/* means we need to back-up to previous sector	*/
	if (sector == 0) {		/* at the first sector			*/
	    sector = tpr->sectors-1;
	} else {
	    sector -= 1;
	}

	for (pos = tpr->sector[sector].size-1; pos >=0 && count; pos--, count--) {
	    if (tpr->sector[sector].data[pos] == 0xff) {
		tpr->sector[sector].data[pos] = 0x00;
	    } else {
		return;			/* return on first non-contiguous 0xff	*/
	    }
	}
    }
}


/************************************************************************
 * NAME:	rnib_selfsync_process()
 *
 * DESCR:	Processes a rnib-encoded floppy image to either put in
 *		or take out encoding for apple self-syncs.
 *		Encoding means turning 0xff's (self-syncs) into 0x00's.
 *		Decoding means turning 0x00's into 0xff's.
 *
 * ARGS:	floppy - the floppy image to process
 *		to_encode - TRUE if encoding, FALSE otherwise
 *		
 *		
 * RETURNS:	nothing
 *
 * NOTES:	- right now, this does a blind replacement
 *		- in the future, it MAY smartly replace only those 0xff's
 *		  that really look like they should be self-syncs.
 ************************************************************************/
void
rnib_selfsync_process(struct floppy *floppy, int to_encode)
{
    int			 track, sector;
    int			 i;
    unsigned char 	*ptr;

    for (track=0; track < floppy->tracks; track++) {
	for (sector = 0; sector < floppy->side[0][track].sectors; sector++) {
	    ptr = floppy->side[0][track].sector[sector].data;
	    for (i=floppy->secsize; i--; ptr++) {
		if (to_encode && (*ptr == (unsigned char)0xd5)) {
		    rnib_selfsync_encode(floppy, track, sector, floppy->secsize-i-1);
		} else if (!to_encode && (*ptr == (unsigned char)0x00)) {
		    *ptr = 0xff;
		}
	    }
	}
    }

}
    
/************************************************************************
 * NAME:	rnib_init()
 *
 * DESCR:	RAW nibble format for apple.  This is an alternative hardware
 *		format for nibble data.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
rnib_init()
{
    if(!floppy_format_register("RNIB", "APPLE 2 Raw NIBble",rnib_guesser, rnib_read, NULL, rnib_report)) {
	return(FALSE);
    }

    if(!floppy_fileext_register("nib",EXT_VERY_SPECIFIC,"RNIB")) {
	return(FALSE);
    }

    if(!floppy_fileext_register("rnib",EXT_VERY_SPECIFIC,"RNIB")) {
	return(FALSE);
    }

    if(!floppy_fileext_register("raw",EXT_VERY_SPECIFIC,"RNIB")) {
	return(FALSE);
    }

    return(TRUE);
}
